
/*
  CLASSiC DAC, Copyright 2013 SILICON CHIP Publications
  DCI.c: functions to use the Data Converter Interface to generate a continuous S/PDIF digital audio stream from a buffered filled with PCM data
  Written by Nicholas Vinen, 2010-2013
*/

#include "DCI.h"
#include "Timer.h"
#include "Biphase.h"
#include "p33Fxxxx.h"
#include <stdlib.h>
#include <string.h>
#include <math.h>

//////////// Buffers, control variables and utility routines ////////////

#define BUFFER_SIZE_WORDS 128
static unsigned short TxBufferA[BUFFER_SIZE_WORDS] __attribute__((space(dma)));
static unsigned short TxBufferB[BUFFER_SIZE_WORDS] __attribute__((space(dma)));

static signed short* DCIBuffer;
static unsigned short DCIBuffer_writepos;
static volatile unsigned short DCIBuffer_readpos;
static unsigned char DCIBuffer_subreadpos;
volatile unsigned char DCIBuffer_full, DCIBuffer_underrun, DCI_WaitingForBufferToFill;
static unsigned char DCI_CurrentBuffer;
static volatile unsigned long DCI_Counter;
static bool DCI_Counting, DCI_Paused;

unsigned char DCI_BitDepth;
unsigned char _DCIBuffer[512*DCI_BUFFER_PAGES];

static bool DCI_Filter_Enabled;
static volatile bool DCI_Filter_Changed;
static unsigned long DCI_Filter_LowCrossoverFrequency, DCI_Filter_HighCrossoverFrequency, DCI_Sampling_Rate;
static signed short DCI_Filter_BassBoost, DCI_Filter_TrebleBoost;

static unsigned long NewSamplingRate;
static signed short NewLPFilterCoeff_A, NewHPFilterCoeff_A, NewLPFilterCoeff_B, NewHPFilterCoeff_B;
static signed short LPFilterCoeff_A, HPFilterCoeff_A, LPFilterCoeff_B, HPFilterCoeff_B, FilterHistory[2], LPFilterHistory[2], HPFilterHistory[2];
static unsigned short NewFilterScale, FilterScale;

static unsigned short ssabs(signed short val) {
  if( val < 0 )
    return -val;
  else
    return val;
}
static unsigned short usmax(unsigned short a, unsigned short b) {
  if( a > b )
    return a;
  else
    return b;
}

//////////// Bass/treble tone control (low-pass/high-pass filter) processing functions ////////////

static void Filter_PreInit(unsigned long sampling_rate, unsigned long low_crossover_freq, unsigned long high_crossover_freq, signed short bass_boost, signed short treble_boost) {
  float W = tan(3.14159265 * low_crossover_freq / sampling_rate);
  float N = 1/(1+W);
  NewSamplingRate = sampling_rate;
  NewLPFilterCoeff_A = (W-1) * N * 32768;
  NewLPFilterCoeff_B =  W    * N * bass_boost;
        W = tan(3.14159265 * high_crossover_freq / sampling_rate);
        N = 1/(1+W);
  NewHPFilterCoeff_A = (W-1) * N * 32768;
  NewHPFilterCoeff_B =         N * treble_boost;
  NewFilterScale = (0 - usmax(ssabs(bass_boost) * 2, ssabs(treble_boost)));
}

static void Filter_Init(unsigned long sampling_rate, unsigned long low_crossover_freq, unsigned long high_crossover_freq, signed short bass_boost, signed short treble_boost) {
  if( sampling_rate != NewSamplingRate )
    Filter_PreInit(sampling_rate, low_crossover_freq, high_crossover_freq, bass_boost, treble_boost);
  LPFilterCoeff_A = NewLPFilterCoeff_A;
  LPFilterCoeff_B = NewLPFilterCoeff_B;
  HPFilterCoeff_A = NewHPFilterCoeff_A;
  HPFilterCoeff_B = NewHPFilterCoeff_B;
  memset(LPFilterHistory, 0, sizeof(LPFilterHistory));
  memset(HPFilterHistory, 0, sizeof(HPFilterHistory));
  FilterScale = NewFilterScale;
}

static unsigned char GetFilterDBAtten(signed short bass_boost, signed short treble_boost) {
  unsigned long RequiredFilterScale = (0 - usmax(ssabs(bass_boost) * 2, ssabs(treble_boost)));
  if( RequiredFilterScale < 16 )
    return 0;
  unsigned char ret = 0;
  while(1) {
    RequiredFilterScale += (RequiredFilterScale * 2573) >> 16;
    if( RequiredFilterScale > 0xFFFF || ++ret == 18 )
      return ret;
  }
}

static void Filter16(signed short* audio_data, unsigned short num_samples) {
  if( DCI_Filter_Changed ) {
    Filter_Init(DCI_Sampling_Rate, DCI_Filter_LowCrossoverFrequency, DCI_Filter_HighCrossoverFrequency, DCI_Filter_BassBoost, DCI_Filter_TrebleBoost);
    DCI_Filter_Changed = false;
  }
  if( DCI_Filter_Enabled && FilterScale ) {
    while( num_samples-- ) {
      signed short data[2] = { audio_data[0], audio_data[1] };
      signed long lp = ((signed long)data[0] + (signed long)FilterHistory[0]) * LPFilterCoeff_B - (signed long)LPFilterHistory[0] * LPFilterCoeff_A;
      signed long hp = ((signed long)data[0] - (signed long)FilterHistory[0]) * HPFilterCoeff_B - (signed long)HPFilterHistory[0] * HPFilterCoeff_A;
      FilterHistory[0] = data[0];
      audio_data[0] = (((signed long)data[0] * (signed long)FilterScale) + lp + hp)>>16;
      LPFilterHistory[0] = lp>>15;
      HPFilterHistory[0] = hp>>15;

                  lp = ((signed long)data[1] + (signed long)FilterHistory[1]) * LPFilterCoeff_B - (signed long)LPFilterHistory[1] * LPFilterCoeff_A;
                  hp = ((signed long)data[1] - (signed long)FilterHistory[1]) * HPFilterCoeff_B - (signed long)HPFilterHistory[1] * HPFilterCoeff_A;
      FilterHistory[1] = data[1];
      audio_data[1] = (((signed long)data[1] * (signed long)FilterScale) + lp + hp)>>16;
	  LPFilterHistory[1] = lp>>15;
	  HPFilterHistory[1] = hp>>15;

      audio_data += 2;
    }
  }
}

static void Filter24(signed short* audio_data, unsigned short num_samples) {
  if( DCI_Filter_Changed ) {
    Filter_Init(DCI_Sampling_Rate, DCI_Filter_LowCrossoverFrequency, DCI_Filter_HighCrossoverFrequency, DCI_Filter_BassBoost, DCI_Filter_TrebleBoost);
    DCI_Filter_Changed = false;
  }
  if( DCI_Filter_Enabled && FilterScale ) {
    while( num_samples-- ) {
      signed short left = ((unsigned char*)audio_data)[1] | (((signed short)((signed char*)audio_data)[2])<<8);
      signed short right;
      signed long lp = ((signed long)left + (signed long)FilterHistory[0]) * LPFilterCoeff_B - (signed long)LPFilterHistory[0] * LPFilterCoeff_A;
      signed long hp = ((signed long)left - (signed long)FilterHistory[0]) * HPFilterCoeff_B - (signed long)HPFilterHistory[0] * HPFilterCoeff_A;
      FilterHistory[0] = left;
      left = (((signed long)left * (signed long)FilterScale) + lp + hp)>>16;
      ((unsigned char*)audio_data)[1] = left;
      ((signed char*)audio_data)[2] = left>>8;
      LPFilterHistory[0] = lp>>16;
	  HPFilterHistory[0] = hp>>16;

      right = audio_data[2];
                  lp = ((signed long)right + (signed long)FilterHistory[1]) * LPFilterCoeff_B - (signed long)LPFilterHistory[1] * LPFilterCoeff_A;
                  hp = ((signed long)right - (signed long)FilterHistory[1]) * HPFilterCoeff_B - (signed long)HPFilterHistory[1] * HPFilterCoeff_A;
      FilterHistory[1] = right;
      audio_data[2] = (((signed long)right * (signed long)FilterScale) + lp + hp)>>16;
	  LPFilterHistory[1] = lp>>16;
	  HPFilterHistory[1] = hp>>16;
      audio_data += 3;
    }
  }
}

unsigned char DCI_Set_Filtering(bool bEnabled, unsigned long LowCrossoverFrequency, unsigned long HighCrossoverFrequency, signed short BassBoost, signed short TrebleBoost, unsigned char FixedScale) {
  asm volatile("disi #32");
  Filter_PreInit(DCI_Sampling_Rate, LowCrossoverFrequency, HighCrossoverFrequency, BassBoost, TrebleBoost);
  DCI_Filter_Enabled = bEnabled;
  DCI_Filter_LowCrossoverFrequency = LowCrossoverFrequency;
  DCI_Filter_HighCrossoverFrequency = HighCrossoverFrequency;
  DCI_Filter_BassBoost = BassBoost;
  DCI_Filter_TrebleBoost = TrebleBoost;
//  DCI_Filter_FixedScale = FixedScale;
  DCI_Filter_Changed = true;
  asm volatile("disi #0");
  return FixedScale ? (bEnabled ? 18 - GetFilterDBAtten(BassBoost, TrebleBoost) : (BassBoost || TrebleBoost ? 16 : 0)) : 0;
}

bool DCI_Get_Filtering(unsigned long* pLowCrossoverFrequency, unsigned long* pHighCrossoverFrequency, signed short* pBassBoost, signed short* pTrebleBoost) {
  if( pLowCrossoverFrequency )
    *pLowCrossoverFrequency = DCI_Filter_LowCrossoverFrequency;
  if( pHighCrossoverFrequency )
    *pHighCrossoverFrequency = DCI_Filter_HighCrossoverFrequency;
  if( pBassBoost )
    *pBassBoost = DCI_Filter_BassBoost;
  if( pTrebleBoost )
    *pTrebleBoost = DCI_Filter_TrebleBoost;
  return DCI_Filter_Enabled;
}

//////////// DCI control functions ////////////

void DCI_Sampling_Rate_Changed(unsigned long rate) {
  DCI_Sampling_Rate = rate;
  DCI_Filter_Changed = true;
}

void DCI_Setup() {
//  int i;
  DCIBuffer = (signed short*)_DCIBuffer;//malloc(2048*2);//512*DCI_BUFFER_PAGES*2);

  // RG13 = data output
  TRISGbits.TRISG13 = 0;
  // RG14 = clock input
  TRISGbits.TRISG14 = 1;

  // set up DCI
  TSCON = 1;
  RSCON = 1;
  DCICON1 = 0;
  DCICON2 = 0;
  DCICON3 = 0;
  DCICON1bits.CSCKD = 1;
  DCICON1bits.CSCKE = 1;
  DCICON1bits.DJST = 1;
  DCICON2bits.WS = 16-1;
  DCICON2bits.BLEN = 1;
  IFS3bits.DCIIF = 0;

  // set up DCI
  DMA0CONbits.DIR = 1;
  DMA0CONbits.MODE = 2;
  DMA0REQbits.IRQSEL = 0x3c;
  DMA0STA = __builtin_dmaoffset(TxBufferA);
  DMA0STB = __builtin_dmaoffset(TxBufferB);
  DMA0PAD = (volatile unsigned int) &TXBUF0;
  DMA0CNT = sizeof(TxBufferA)/2-1;
  IFS0bits.DMA0IF = 0;
  IEC0bits.DMA0IE = 1;
}

void __attribute__((__interrupt__,no_auto_psv)) _DCIInterrupt(void) {
  if( DCI_Counting ) {
    ++DCI_Counter;
    TXBUF0 = 0;
    TXBUF1 = 0;
  }
  IFS3bits.DCIIF = 0;
}

unsigned long DCI_Measure_Clock_Frequency(unsigned short Period_ms) {
  IEC3bits.DCIIE = 1;
  DCI_Counter = 0;
  TXBUF0 = 0;
  TXBUF1 = 0;
  DCI_Counting = true;
  DCICON2bits.BLEN = 1;
  DCICON1bits.DCIEN = 1;
  Timer_Delay_ms(Period_ms);
  DCICON1bits.DCIEN = 0;
  DCI_Counting = false;
  IEC3bits.DCIIE = 0;

  return DCI_Counter * 1000UL / Period_ms * 32;
}

static const signed short DCI_zero[BUFFER_SIZE_WORDS/4] = { 0 };

//////////// Interrupt handler to feed data to DCI using DMA buffers ////////////

void __attribute__((__interrupt__,auto_psv)) _DMA0Interrupt(void) {
  unsigned short* buf = DCI_CurrentBuffer ? TxBufferB : TxBufferA;
  signed short* src = DCIBuffer+DCIBuffer_readpos+DCIBuffer_subreadpos;
  DCI_CurrentBuffer ^= 1;
  IFS0bits.DMA0IF = 0;

  if( ((DCI_Paused || DCI_WaitingForBufferToFill) && !DCIBuffer_subreadpos) ) {
    Biphase_Encode_16bit(buf, DCI_zero, BUFFER_SIZE_WORDS/8);
    DCI_Filter_Changed = true;
    if( DCI_WaitingForBufferToFill && DCIBuffer_full )
      DCI_WaitingForBufferToFill = false;
  } else {
    if( PORTFbits.RF4 /* SD card has been removed */ ) {
      Biphase_Encode_16bit(buf, DCI_zero, BUFFER_SIZE_WORDS/8);
    } else if( DCI_BitDepth == 24 ) {
      if( DCI_Filter_Enabled && DCI_Sampling_Rate <= 48000 )
        Filter24(src, BUFFER_SIZE_WORDS/8);
      Biphase_Encode_24bit(buf, src, BUFFER_SIZE_WORDS/8);
    } else {
      if( DCI_Filter_Enabled && DCI_Sampling_Rate <= 48000 )
        Filter16(src, BUFFER_SIZE_WORDS/8);
      Biphase_Encode_16bit(buf, src, BUFFER_SIZE_WORDS/8);
    }

    if( DCIBuffer_readpos == DCIBuffer_writepos && !DCIBuffer_full )
      DCIBuffer_underrun = 1;

    if( DCI_BitDepth == 24 ) {
      DCIBuffer_subreadpos += (BUFFER_SIZE_WORDS/8)*3;
      if( DCIBuffer_subreadpos < (BUFFER_SIZE_WORDS/8)*3 ) {
        DCIBuffer_readpos += 256;
        if( DCIBuffer_readpos == 256*DCI_BUFFER_PAGES ) {
          DCIBuffer_readpos = 0;
          DCIBuffer_subreadpos = 0; // should have rolled over to zero anyway but this is just in case it doesn't
        }
        DCIBuffer_full = 0;
      }
    } else {
      DCIBuffer_subreadpos += BUFFER_SIZE_WORDS/4;
      if( DCIBuffer_subreadpos == 0 ) {
        DCIBuffer_readpos += 256;
        if( DCIBuffer_readpos == 256*DCI_BUFFER_PAGES )
          DCIBuffer_readpos = 0;
        DCIBuffer_full = 0;
      }
    }
  }
}

void DCI_Start_SPDIF_Transmission() {
  DMA0CONbits.CHEN = 1;
  DCICON2bits.BLEN = 0;
  DCICON1bits.DCIEN = 1;
  TXBUF0 = 0;
}

//////////// Functions to manage the intermediate buffers for the DCI ////////////

signed short* DCI_Prepare_Write_2_Channel() {
  while( DCIBuffer_full )
    ;
  return DCIBuffer+DCIBuffer_writepos;
}

signed short* DCI_Prepare_Write_2_Channel_Size(unsigned char* NumBlocks) {
  signed short BufferSpace;

  while( DCIBuffer_full )
    ;

  if( DCIBuffer_writepos/256 + *NumBlocks > DCI_BUFFER_PAGES )
    *NumBlocks = DCI_BUFFER_PAGES - DCIBuffer_writepos/256;

  while(1) {
    BufferSpace = DCIBuffer_readpos - DCIBuffer_writepos;
    if( BufferSpace < 0 )
      BufferSpace += 256 * DCI_BUFFER_PAGES;
    if( BufferSpace == 0 || BufferSpace >= 256 * *NumBlocks )
      break;
    DCI_WaitingForBufferToFill = 0;
  }

  return DCIBuffer+DCIBuffer_writepos;
}

void DCI_Commit_Write_2_Channel() {
  asm volatile("disi #16");
  DCIBuffer_writepos += 256;
  if( DCIBuffer_writepos == 256*DCI_BUFFER_PAGES )
    DCIBuffer_writepos = 0;
  if( DCIBuffer_writepos == DCIBuffer_readpos )
    DCIBuffer_full = 1;
  asm volatile("disi #0");
}

void DCI_Commit_Write_2_Channel_Size(unsigned char NumBlocks) {
  asm volatile("disi #16");
  DCIBuffer_writepos += 256 * NumBlocks;
  if( DCIBuffer_writepos == 256*DCI_BUFFER_PAGES )
    DCIBuffer_writepos = 0;
  if( DCIBuffer_writepos == DCIBuffer_readpos )
    DCIBuffer_full = 1;
  asm volatile("disi #0");
}

void DCI_Write(const signed short* data, unsigned char num_channels) {
  while( DCIBuffer_full )
    ;
  if( num_channels == 1 ) {
    signed short* dest = DCIBuffer+DCIBuffer_writepos;
    unsigned short i = 128;
    while( i-- ) {
      dest[0] = dest[1] = data[0];
      dest += 2;
      data += 1;
    }
  } else {
    memcpy(DCIBuffer+DCIBuffer_writepos, data, 512);
  }
  asm volatile("disi #16");
  DCIBuffer_writepos += 256;
  if( DCIBuffer_writepos == 256*DCI_BUFFER_PAGES )
    DCIBuffer_writepos = 0;
  if( DCIBuffer_writepos == DCIBuffer_readpos )
    DCIBuffer_full = 1;
  asm volatile("disi #0");
}

void DCI_Pause(bool Paused) {
  if( DCI_Paused && !Paused )
    DCI_WaitingForBufferToFill = true;
  DCI_Paused = Paused;
}

void DCI_Clear_Buffer() {
  memset(DCIBuffer, 0, 512*DCI_BUFFER_PAGES);
}

void DCI_Reset_Buffer() {
  DCIBuffer_writepos = DCIBuffer_readpos = DCIBuffer_subreadpos = 0;
  DCIBuffer_full = 0;
}

unsigned char* DCI_Get_Buffer() {
  return (unsigned char*)DCIBuffer;
}

bool DCI_Is_Running() {
  return DMA0CONbits.CHEN && !DCI_Paused;
}

bool DCI_Write_Will_Be_on_24bit_Boundary() {
  return (DCIBuffer_writepos%3) == 0;
}
